package de.accxia.jira.addon.IUM.job;


import com.atlassian.jira.application.ApplicationRoleManager;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.atlassian.plugin.spring.scanner.annotation.component.Scanned;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.status.JobDetails;
import de.accxia.jira.addon.IUM.config.DAO;
import de.accxia.jira.addon.IUM.model.JobResult;
import de.accxia.jira.addon.IUM.model.JobResultDTO;
import de.accxia.jira.addon.IUM.repository.JobResultRepository;
import de.accxia.jira.addon.IUM.repository.PocketRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * GET/POST
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/config
 *
 *
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/start
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/stop
 *
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/runnow
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/jobcheck
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/jobresults
 * https://aurelian-jira.accxia.com/rest/IUM/latest/sjob/jobtest
 */
@Path("/sjob")
@Consumes({ MediaType.TEXT_HTML, MediaType.APPLICATION_JSON})
@Produces({ MediaType.APPLICATION_JSON })
@Scanned
public class JobSyncRestController {
    private static final Logger LOG = LoggerFactory.getLogger(JobSyncRestController.class);

    @ComponentImport
    private final JiraAuthenticationContext jiraAuthenticationContext;
    private final MonitorJobRunner monitorJobRunner;
    private final PocketRepository pocketRepository;
    private final JobResultRepository jobResultRepository;
    private final ApplicationRoleManager applicationRoleManager;

    @Inject
    public JobSyncRestController(MonitorJobRunner monitorJobRunner, PocketRepository pocketRepository, JobResultRepository jobResultRepository,
                                 ApplicationRoleManager applicationRoleManager, JiraAuthenticationContext jiraAuthenticationContext) {
        this.monitorJobRunner=monitorJobRunner;
        this.pocketRepository = pocketRepository;
        this.jobResultRepository=jobResultRepository;
        this.applicationRoleManager=applicationRoleManager;
        this.jiraAuthenticationContext=jiraAuthenticationContext;
    }

    @Path("/config")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response getCurrentConfig() {
        Map<String,String> map = new HashMap<>();
        map.put(JobData.REPEAT_INTERVAL, DAO.getRepeatInterval());
        map.put(JobData.STARTING_FROM, DAO.getStartingFrom());
        map.put(JobData.ENABLE_GROUPS,DAO.getIUMGroups());
        map.put(JobData.DISABLE_GROUPS,DAO.getIUMGroupsDisabled());

        map.put("working",String.valueOf(monitorJobRunner.isWorking(IUMSyncJob.class)));
        if(monitorJobRunner.getJobDetails(IUMSyncJob.class)!=null){
            map.put("jobDetails",convert2String(monitorJobRunner.getJobDetails(IUMSyncJob.class)));
        }
        List<JobDetails> details=monitorJobRunner.getJobsByJobRunnerKey(IUMSyncJob.class);
        if(details!=null){
            for (int i = 0; i < details.size(); i++) {
                map.put("jobDetails"+i,details.get(i).toString());
            }
        }

        if(monitorJobRunner.isWorking(IUMSyncJob.class)){
            map.put("lastRunForJob",convert2String(monitorJobRunner.getLastRunForJob(IUMSyncJob.class)));
            map.put("lastSuccessfulRunForJob",convert2String(monitorJobRunner.getLastSuccessfulRunForJob(IUMSyncJob.class)));
            map.put("nextRunDate",convert2String(monitorJobRunner.getNextRunDate(IUMSyncJob.class)));
        }


        return Response.ok(map).build();
    }

    @Path("/config")
    @POST
    @Produces({ MediaType.APPLICATION_JSON })
    public Response setCurrentConfig(@QueryParam(JobData.REPEAT_INTERVAL) String repeatinterval,
                                     @QueryParam(JobData.STARTING_FROM) String startingFrom) {

        DAO.updateRepeatInterval(repeatinterval);
        DAO.updateStartingFrom(startingFrom);
        return getCurrentConfig();
    }

    @Path("/start")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response startDebugJob() {
        try{
            monitorJobRunner.doUnSchedule(IUMSyncJob.class);
            //jobKey = monitorJobRunner.doSchedule(JobDataUtils.createIUMJobData(),IUMJob.getInstance());
            //create fake schedule with running at 10 sec

            IUMSyncJob.getInstance().injectService(pocketRepository,jobResultRepository);
            IUMSyncJob.getInstance().injectService(applicationRoleManager);
            String jobKey = monitorJobRunner.doSchedule(JobDataUtils.createFakeIUMSyncJobData(),IUMSyncJob.getInstance());

            if (LOG.isDebugEnabled()) {
                LOG.debug("jobKey= " + jobKey);
            }

            return Response.ok("success").build();
        } catch (SchedulerServiceException e) {
            return Response.status(Response.Status.BAD_REQUEST).entity(e.getMessage()).build();
        }
    }

    @Path("/stop")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response stopDebugJob() {
        try{
            monitorJobRunner.doUnSchedule(IUMSyncJob.class);
            return Response.ok("success").build();
        } catch (SchedulerServiceException e) {
            return Response.status(Response.Status.BAD_REQUEST).entity(e.getMessage()).build();
        }
    }

    private static LocalThread localThread=null;


    @Path("/runnow")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response runNowsJob() {
        if (JobSyncRestController.localThread != null && JobSyncRestController.localThread.isWorking ) {
            String response = "{\"message\":\"processing\"}";
            return Response.ok(response).build();
        }

        try {
            long duration = Long.parseLong(DAO.getInactivityDuration()) * 60 * 1000;

            Map<String, Serializable> parameters = new HashMap<>();
            parameters.put(JobData.SYNC_SOFTWARE_SRC_GROUP, DAO.getSyncSoftwareSrcGoups());
            parameters.put(JobData.SYNC_SOFTWARE_TARGET_GROUP, DAO.getSyncSoftwareTargetGoups());
            parameters.put(JobData.SYNC_SDESK_SRC_GROUP, DAO.getSyncSDeskSrcGoups());
            parameters.put(JobData.SYNC_SDESK_TARGET_GROUP, DAO.getSyncSDeskTargetGoups());

            parameters.put(JobData.USER_INACTIVITY_INTERVAL,duration);
            final String currentUser = jiraAuthenticationContext.getLoggedInUser() !=null ? jiraAuthenticationContext.getLoggedInUser().getUsername()  : "IUMPlugin";

            parameters.put(JobData.CURRENT_USER,currentUser );


            JobSyncRestController.localThread = new LocalThread(currentUser, parameters);
            JobSyncRestController.localThread.start();



            String response = "{\"message\":\"processing\"}";
            return Response.ok(response).build();

        } catch (Exception e) {
            String response = "{\"message\":\"" + e.getMessage() + "\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
        }
    }

    @Path("/jobcheck")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response runNowsCheckJob() {
        if (JobSyncRestController.localThread  == null) {
            String response = "{\"message\":\"not stating\"}";
            return Response.ok(response).build();
        }

        if (JobSyncRestController.localThread.isWorking ) {
            String response = "{\"message\":\"processing\"}";
            return Response.ok(response).build();
        }

        try {

            if (JobSyncRestController.localThread.errorMessage !=null ) {
                String response = "{\"message\":\""+JobSyncRestController.localThread.errorMessage+"\"}";
                return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
            }

            return Response.ok(JobSyncRestController.localThread.processResultList).build();

        } catch (Exception e) {
            String response = "{\"message\":\"" + e.getMessage() + "\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
        }
    }

    @Path("/jobresults")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response getLatestJobResults() {
        if(this.jobResultRepository == null){
            String response = "{\"message\":\"not stating\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
        }

        List<JobResultDTO> jobResultDTOList = Arrays.stream(this.jobResultRepository.findAll(0, 10))
                        .map(jobResult -> JobResultDTO.fromAO(jobResult))
                        .collect(Collectors.toList());

        return Response.ok(jobResultDTOList).build();
    }

    @Path("/jobtest")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response getTestJobResults(@QueryParam(JobResult.PROCESS) String process){
        if(this.jobResultRepository == null){
            String response = "{\"message\":\"not stating\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
        }

        List<JobResultDTO> jobResultDTOList = Arrays.stream(this.jobResultRepository.findAll(new String[]{process}, 0, 10))
                .map(jobResult -> JobResultDTO.fromAO(jobResult))
                .collect(Collectors.toList());

        return Response.ok(jobResultDTOList).build();
    }

    class  LocalThread extends Thread {
        Map<String, Serializable> parameters;
        String currentUser;
        List<ProcessResult> processResultList=null;

        String errorMessage;
        boolean isWorking;

        public LocalThread(String currentUser,Map<String, Serializable> parameters){
            this.parameters=parameters;
            this.currentUser=currentUser;
            this.isWorking=false;
            this.errorMessage=null;
        }

        @Override
        public void run() {
            try {
                long startTime = System.currentTimeMillis();

                JobSyncRestController.localThread.isWorking=true;
                IUMSyncJob.getInstance().injectService(pocketRepository,jobResultRepository);
                IUMSyncJob.getInstance().injectService(applicationRoleManager);

                parameters.put(JobData.CURRENT_USER, currentUser );
                List<ProcessResult> processResultList = IUMSyncJob.getInstance().doProcessingJob(parameters);

                long stopTime = System.currentTimeMillis();
                long elapsedTime = stopTime - startTime;

                JobSyncRestController.localThread.processResultList=processResultList;
                doSaveProcessResultList(processResultList,parameters,elapsedTime);

                //JobSyncRestController.localThread.isWorking=false;
                //sendNotification(currentUser.getName(), "Automatic User Sync Job", "success");
            } catch (Exception e) {
                LOG.error("Exception: " + e.getMessage(), e);
                try {
                    JobSyncRestController.localThread.errorMessage = e.getMessage();
                    //sendNotification(currentUser.getName(), "Automatic User Sync Job", e.getMessage());
                } catch (Exception ex) {
                    LOG.error("Exception: " + ex.getMessage(), ex);
                    throw new RuntimeException(ex);
                }
            }finally {
                JobSyncRestController.localThread.isWorking=false;
            }
        }
    }

    private void doSaveProcessResultList(List<ProcessResult> processResultList,Map<String, Serializable> parameters, long elapsedTime) {
        final String currentUser = (String)parameters.get(JobData.CURRENT_USER);
        for (ProcessResult processResult:processResultList){
            jobResultRepository.save(JobResultDTO.createJobResult(processResult,currentUser,elapsedTime));
        }
    }

    private <T> String convert2String(T t) {
        return (t!= null) ? t.toString() : "";

    }
}


