package de.accxia.jira.addon.IUM.servlet.filter;

import com.atlassian.adapter.jackson.ObjectMapper;
import com.atlassian.jira.avatar.Avatar;
import com.atlassian.jira.avatar.AvatarService;
import com.atlassian.jira.avatar.AvatarUrls;
import com.atlassian.jira.bc.user.search.UserSearchParams;
import com.atlassian.jira.bc.user.search.UserSearchService;
import com.atlassian.jira.bc.user.search.UserSearchUtilities;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.IssueManager;
import com.atlassian.jira.permission.ProjectPermissions;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.rest.v2.issue.*;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.security.plugin.ProjectPermissionKey;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.user.ApplicationUsers;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.jira.util.DelimeterInserter;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.plugin.spring.scanner.annotation.component.Scanned;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.component.ComponentLocator;
import com.opensymphony.util.TextUtils;

import de.accxia.jira.addon.IUM.search.UserResults;
import org.apache.commons.lang.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.UriBuilder;
import java.io.IOException;
import java.net.URI;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Scanned
public class IUMRestFilter  implements Filter {

    private static final Logger LOG = LoggerFactory.getLogger(IUMRestFilter.class);

    public static final String  ASSIGNABLE = "/rest/api/[\\d\\w]+/user/assignable/search";
    //https://aurelian-jira.accxia.com/rest/internal/2/users/assignee?issueKey=TS-3&maxResults=100&query=c&_=1714397430189
    public static final String  ASSIGNEE = "/rest/internal/[\\d\\w]+/users/assignee";
    public static final String  MULTI_ASSIGNABLE = "/rest/api/[\\d\\w]+/user/assignable/multiProjectSearch";

    // https://aurelian-jira.accxia.com/rest/api/2/search/user?username=ac&project=TA&maxResults=50&startAt=0
    public static final String  USER = "/rest/api/[\\d\\w]+/search/user";
    public static final String  AGENTS = "/rest/servicedesk/([\\d\\w]+)/pages/people/agents/([\\d\\w]+)/search";
    public static final String  AUTOMATION = "/rest/servicedesk/automation/[\\d\\w]+/components/then/([\\d\\w]+)/validate";
    public static final String  AUTOMATION_VALIDATE = "/rest/servicedesk/automation/[\\d\\w]+/ruleset/[\\d\\w]+/expectedRevisionId/([\\d\\w]+(/validate)?)";

    // https://aurelian-jira.accxia.com/rest/api/1.0/users/picker/filter?issueKey=SAU-1&showAvatar=true&query=acco&_=1727247486457
    public static final String  WATCHERS = "/rest/api/[\\d\\.\\w]+/users/picker/filter";


    //\{\\"assignee\\":\\"([\w\d]+)\\"\}
    public static final String  FIELDVALUE_ASSIGNABLE ="\\{\\\\\"assignee\\\\\":\\\\\"([\\w\\d]+)\\\\\"\\}";

    public static final Pattern PATTERN_ASSIGNABLE = Pattern.compile(ASSIGNABLE);
    public static final Pattern PATTERN_ASSIGNEE = Pattern.compile(ASSIGNEE);
    public static final Pattern PATTERN_MULTI_ASSIGNABLE = Pattern.compile(MULTI_ASSIGNABLE);
    public static final Pattern PATTERN_USER = Pattern.compile(USER);
    public static final Pattern PATTERN_WATCHERS = Pattern.compile(WATCHERS);
    public static final Pattern PATTERN_AGENTS = Pattern.compile(AGENTS);
    public static final Pattern PATTERN_AUTOMATION = Pattern.compile(AUTOMATION);
    public static final Pattern PATTERN_AUTOMATION_VALIDATE = Pattern.compile(AUTOMATION_VALIDATE);

    public static final Pattern PATTERN_FIELDVALUE_ASSIGNABLE= Pattern.compile(FIELDVALUE_ASSIGNABLE);
    private final UserManager userManager;
    private final UserSearchService userSearchService;
    @ComponentImport
    private final ApplicationProperties applicationProperties;
    @ComponentImport
    private final ProjectManager projectManager;
    @ComponentImport
    private final PermissionManager permissionManager;

    @ComponentImport
    private final IssueManager issueManager;
    @ComponentImport
    private final AvatarService avatarService;
    @ComponentImport
    private final I18nHelper i18n;

    @Inject
    public IUMRestFilter(ApplicationProperties applicationProperties, ProjectManager projectManager, PermissionManager permissionManager, IssueManager issueManager, AvatarService avatarService, I18nHelper i18n) {
        this.permissionManager = permissionManager;
        this.issueManager = issueManager;
        this.avatarService = avatarService;
        this.i18n = i18n;
        this.userManager=ComponentAccessor.getUserManager();
        this.userSearchService = ComponentLocator.getComponent(UserSearchService.class);
        this.applicationProperties = applicationProperties;
        this.projectManager = projectManager;

    }


    /**
     * Does nothing but prints INFO log message
     *
     * @param filterConfig not used
     * @throws ServletException not thrown
     */
    public void init(final FilterConfig filterConfig) throws ServletException    {
        LOG.info("IUMRestFilter initialized.");
    }

    // https://aurelian-jira.accxia.com/rest/internal/2/users/assignee?issueKey=TS-3&maxResults=100&query=cr&_=1714397430190

    // https://aurelian-jira.accxia.com/rest/api/2/user/assignable/search?username=ac&project=TA&maxResults=50&startAt=0
    // https://aurelian-jira.accxia.com/rest/api/latest/user/assignable/multiProjectSearch?username=xa&projectKeys=TA&maxResults=100&_=1714298078606
    // https://aurelian-jira.accxia.com/rest/api/latest/user/assignable/search?username=ac&project=TA&maxResults=50&startAt=0
    // https://aurelian-jira.accxia.com/rest/servicedesk/1/pages/people/agents/TA/search?query=&_=1714113447736
    //
    // https://aurelian-jira.accxia.com/rest/servicedesk/automation/1/components/then/TA/validate
    // https://aurelian-jira.accxia.com/rest/servicedesk/automation/1/ruleset/91/expectedRevisionId/91
    // https://aurelian-jira.accxia.com/rest/servicedesk/automation/1/ruleset/91/expectedRevisionId/91/validate
    /**
     *
     * @param servletRequest request
     * @param servletResponse response
     * @param filterChain filter chain
     * @throws IOException if another filter in the filter chain throws it
     * @throws ServletException if another filter in the filter chain throws it
     */
    public void doFilter(final ServletRequest servletRequest, final ServletResponse servletResponse, final FilterChain filterChain)
            throws IOException, ServletException{

        // now look into the request and log it
        final HttpServletRequest httpServletRequest = (HttpServletRequest) servletRequest;
        final HttpServletResponse httpServletResponse = (HttpServletResponse) servletResponse;

        try {

            if(StringUtils.isEmpty(httpServletRequest.getRequestURI())){
                filterChain.doFilter(httpServletRequest, httpServletResponse);
                return;
            }

            //1.  request like : /user/assignable/search
            if (PATTERN_ASSIGNABLE.matcher(httpServletRequest.getRequestURI()).matches()){
                doSearchAssignable(httpServletRequest,httpServletResponse);
                return;
            }
            //2. request like : /user/assignable/multiProjectSearch
            if (PATTERN_MULTI_ASSIGNABLE.matcher(httpServletRequest.getRequestURI()).matches()){
                doSearchAssignable(httpServletRequest,httpServletResponse);
                return;
            }
            //2. request like : /users/assignee
            if (PATTERN_ASSIGNEE.matcher(httpServletRequest.getRequestURI()).matches()){
                doSearchAssignee(httpServletRequest,httpServletResponse);
                return;
            }

            //3.  request like : /search/user
            if (PATTERN_USER.matcher(httpServletRequest.getRequestURI()).matches()){
                doSearchUser(httpServletRequest,httpServletResponse);
                return;
            }


            //3.  request like : /rest/api/1.0/users/picker/filter
            if (PATTERN_WATCHERS.matcher(httpServletRequest.getRequestURI()).matches()){
                doSearchWatchers(httpServletRequest,httpServletResponse);
                return;
            }
            /*
            //4.  request like : /pages/people/agents/TA/search
            Matcher matcherAgents = PATTERN_AGENTS.matcher(httpServletRequest.getRequestURI());
            if (matcherAgents.matches()){
                String projectKey =  matcherAgents.groupCount()>=2 ? matcherAgents.group(2):"";
                String query = httpServletRequest.getParameter("query");
                final ContentCachingResponseWrapper responseWrapper = new ContentCachingResponseWrapper((HttpServletResponse) servletResponse);

                filterChain.doFilter(servletRequest, responseWrapper);
                doAddAgents(projectKey,query,responseWrapper,servletResponse);
                return;
            }
            */

            filterChain.doFilter(httpServletRequest, httpServletResponse);

        } catch (Exception e) {
            LOG.error("Exception: " + e.getMessage(), e);
        }

    }


    private ApplicationUser extractAssigneeFromValidateContentBody(String contentBody) {
        ApplicationUser assignee = null;
        try {
            Matcher fieldValueAssignable = PATTERN_FIELDVALUE_ASSIGNABLE.matcher(contentBody);
            if (fieldValueAssignable.find()){
                String user = fieldValueAssignable.groupCount()>=1 ? fieldValueAssignable.group(1):"";

                assignee = this.userManager.getUserByName(user);
            }

        }catch (Exception e){
            LOG.error("Exception: " + e.getMessage());
        }

        return assignee;
    }


    // https://aurelian-jira.accxia.com/rest/api/2/user/assignable/search?username=ac&project=TA&maxResults=50&startAt=0
    // https://aurelian-jira.accxia.com/rest/api/latest/user/assignable/search?username=acc&projectKeys=AD&issueKey=AD-25&actionDescriptorId=21&maxResults=100&_=1710532214245
    //
    // https://aurelian-jira.accxia.com/rest/api/latest/user/assignable/multiProjectSearch?username=xa&projectKeys=TA&maxResults=100&_=1714298078606

    private void doSearchAssignable(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException {
        try {
            String uname=httpServletRequest.getParameter("username");
            String project1=httpServletRequest.getParameter("project");
            String project2=httpServletRequest.getParameter("projectKeys");
            String projectKeys = project1 != null ? project1 :project2;
            String username=uname!=null ? uname :"";

            String maxResults=httpServletRequest.getParameter("maxResults");
            Project project = projectManager.getProjectByCurrentKey(projectKeys);

            UserSearchParams.Builder builder = new UserSearchParams.Builder()
                    .allowEmptyQuery(true);
            if(project!=null){
                builder.filterByProjectIds(Arrays.asList(project.getId()));
            }

            UserSearchParams userSearchParams =builder
                    .includeActive(true)
                    .includeInactive(false)
                    .maxResults(100)
                    .build();


            Collection<ApplicationUser> users = this.userSearchService.findUsers(username, userSearchParams);

            //filter
            users = filterToProject(users,project,ProjectPermissions.ASSIGNABLE_USER);

            //convert
            List<UserBean> userBeanList = this.convertUsersToUserBeans(Locale.getDefault(), users);
            ObjectMapper objectMapper = new ObjectMapper();

            // Convert list to JSON string
            String jsonString = objectMapper.writeValueAsString(userBeanList);

            // Set content type to JSON
            httpServletResponse.setContentType("application/json");

            // Write JSON response to output stream

            httpServletResponse.getWriter().write(jsonString);
            return;
        } catch (Exception e) {
            LOG.error("Exception:" + e.getMessage(), e);
            // Set error status code
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);

            // Set content type to plain text or JSON, depending on your application
            httpServletResponse.setContentType("text/plain");

            // Write error message to response
            httpServletResponse.getWriter().write("An error occurred: " + e.getMessage());

        }



    }


    private void doSearchAssignee(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException {
        try {
            String issueKey=httpServletRequest.getParameter("issueKey");
            String query=httpServletRequest.getParameter("query");
            String maxResults=httpServletRequest.getParameter("maxResults");

            Issue issue= issueManager.getIssueObject(issueKey);
            Project project =issue.getProjectObject();

            UserSearchParams.Builder builder = new UserSearchParams.Builder()
                    .ignorePermissionCheck(false)
                    .allowEmptyQuery(true);

            if(project!=null){
                builder.filterByProjectIds(Arrays.asList(project.getId()));
            }

            UserSearchParams userSearchParams =builder
                    .includeActive(true)
                    .includeInactive(false)
                    .maxResults(100)
                    .build();

            Collection<ApplicationUser> users = this.userSearchService.findUsers(query, userSearchParams);

            //filter
            users = filterToProject(users,project,ProjectPermissions.ASSIGNABLE_USER);

            //convert
            List<UserBean> userBeanList = this.convertUsersToUserBeans(Locale.getDefault(), users);

            ObjectMapper objectMapper = new ObjectMapper();

            // Convert list to JSON string
            String jsonString = objectMapper.writeValueAsString(new UserResults(userBeanList));

            // Set content type to JSON
            httpServletResponse.setContentType("application/json");

            // Write JSON response to output stream

            httpServletResponse.getWriter().write(jsonString);
            return;
        } catch (Exception e) {
            LOG.error("Exception:" + e.getMessage(), e);
            // Set error status code
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);

            // Set content type to plain text or JSON, depending on your application
            httpServletResponse.setContentType("text/plain");

            // Write error message to response
            httpServletResponse.getWriter().write("An error occurred: " + e.getMessage());

        }



    }

    private void doSearchUser(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException {
        try {
            String username=httpServletRequest.getParameter("username");

            String project1=httpServletRequest.getParameter("project");
            String project2=httpServletRequest.getParameter("projectKeys");
            String projectKeys = project1 != null ? project1 :project2;

            String maxResults=httpServletRequest.getParameter("maxResults");
            Project project = projectManager.getProjectByCurrentKey(projectKeys);

            UserSearchParams.Builder builder = new UserSearchParams.Builder()
                    .allowEmptyQuery(true);
            if(project!=null){
                builder.filterByProjectIds(Arrays.asList(project.getId()));
            }


            UserSearchParams userSearchParams =builder
                    .includeActive(true)
                    .includeInactive(true)
                    .maxResults(100)
                    .build();

            Collection<ApplicationUser> users = this.userSearchService.findUsers(username, userSearchParams);

            //filter
            users = filterToProject(users,project,ProjectPermissions.ASSIGNABLE_USER);

            //convert
            List<UserBean> userBeanList = this.convertUsersToUserBeans(Locale.getDefault(), users);
            ObjectMapper objectMapper = new ObjectMapper();

            // Convert list to JSON string
            String jsonString = objectMapper.writeValueAsString(userBeanList);

            // Set content type to JSON
            httpServletResponse.setContentType("application/json");

            // Write JSON response to output stream

            httpServletResponse.getWriter().write(jsonString);
            return;
        } catch (Exception e) {
            LOG.error("Exception:" + e.getMessage(), e);
            // Set error status code
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);

            // Set content type to plain text or JSON, depending on your application
            httpServletResponse.setContentType("text/plain");

            // Write error message to response
            httpServletResponse.getWriter().write("An error occurred: " + e.getMessage());

        }



    }

    private void doSearchWatchers(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException {
        try {
            //issueKey = SAU-1
            String issueKey=httpServletRequest.getParameter("issueKey");
            String query=httpServletRequest.getParameter("query");
            String showAvatar=httpServletRequest.getParameter("showAvatar");

            String projectKeys=getProjectkeyFromIssuekey(issueKey);
            Project project = projectManager.getProjectByCurrentKey(projectKeys);

            UserSearchParams.Builder builder = new UserSearchParams.Builder()
                    .allowEmptyQuery(true);
            if(project!=null){
                builder.filterByProjectIds(Arrays.asList(project.getId()));
            }


            UserSearchParams userSearchParams =builder
                    .includeActive(true)
                    .includeInactive(true)
                    .maxResults(100)
                    .build();

            Collection<ApplicationUser> users = this.userSearchService.findUsers(query, userSearchParams);

            //filter
            users = filterToProject(users,project,ProjectPermissions.VIEW_VOTERS_AND_WATCHERS);

            //convert
            UserPickerResultsBean userPickerResultsBean = this.convertUsersToUserPickerUser(query,true, users);
            ObjectMapper objectMapper = new ObjectMapper();

            // Convert list to JSON string
            String jsonString = objectMapper.writeValueAsString(userPickerResultsBean);

            // Set content type to JSON
            httpServletResponse.setContentType("application/json");

            // Write JSON response to output stream

            httpServletResponse.getWriter().write(jsonString);
            return;
        } catch (Exception e) {
            LOG.error("Exception:" + e.getMessage(), e);
            // Set error status code
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);

            // Set content type to plain text or JSON, depending on your application
            httpServletResponse.setContentType("text/plain");

            // Write error message to response
            httpServletResponse.getWriter().write("An error occurred: " + e.getMessage());

        }



    }


    @Override
    public void destroy() {

    }

    private Collection<ApplicationUser> filterToProject(Collection<ApplicationUser> users, Project project, ProjectPermissionKey projectPermissionKey){
        if(project!=null){
            return  users.stream()
                    .filter(applicationUser -> permissionManager.hasPermission(projectPermissionKey, project, (ApplicationUser) applicationUser))
                    .collect(Collectors.toList());
        }
        return  users;
}

    private List<UserBean> convertUsersToUserBeans(Locale aDefault, Collection<ApplicationUser> users)  {
        List<UserBean> userBeanList = new ArrayList<>();
        for (ApplicationUser user:users){
            userBeanList.add(new UserBean(
                    createSelfLink(user),
                    user.getKey(),
                    user.getName(),
                    user.getDisplayName(),
                    true,
                    user.getEmailAddress(),
                    null,
                    getAvatarURLs(user),
                    null
            ));
        }

        return userBeanList;
    }

    private UserPickerResultsBean convertUsersToUserPickerUser(String query, boolean showAvatar, Collection<ApplicationUser> users)  {
        //TODO
        boolean canShowEmailAddresses= true;
        int total = users.size();
        int showing=0;

        List<UserPickerUser> userPickerUserList = new ArrayList<>();
        for (ApplicationUser user:users){

            String html = this.formatUser(user, query, canShowEmailAddresses);
            userPickerUserList.add(new UserPickerUser(user.getName(), ApplicationUsers.getKeyFor(user), user.getDisplayName(), html,
                    showAvatar ? this.avatarService.getAvatarURL(user, user.getName(), Avatar.Size.SMALL) : null));
        }

        return new UserPickerResultsBean(userPickerUserList, this.i18n.getText("jira.ajax.autocomplete.user.more.results",
                String.valueOf(showing), String.valueOf(total)), total);
    }

    private static Map<String, URI> getAvatarURLs(ApplicationUser forUser) {
        Avatar avatar = ComponentAccessor.getAvatarService().getAvatar(forUser, forUser);
        return avatar != null ? AvatarUrls.getAvatarURLs(forUser, avatar) : null;
    }
    private URI createSelfLink(ApplicationUser forUser) {
        UriBuilder urlBuilder = UriBuilder.fromPath(this.applicationProperties.getBaseUrl())
                .path("rest")
                .path(UserResource.class).queryParam("username", new Object[]{"{0}"});

        return urlBuilder.build(new Object[]{forUser.getUsername()});
    }

    private String formatUser(ApplicationUser user, String query, boolean canShoweEmailAddresses) {
        DelimeterInserter delimeterInserter = new DelimeterInserter("<strong>", "</strong>");
        delimeterInserter.setConsideredWhitespace(UserSearchUtilities.SEPARATORS_STRING);
        String[] terms = new String[]{query};
        String userFullName = delimeterInserter.insert(TextUtils.htmlEncode(user.getDisplayName()), terms);
        String userName = delimeterInserter.insert(TextUtils.htmlEncode(user.getName()), terms);
        StringBuilder sb = new StringBuilder();
        sb.append(userFullName);
        if (canShoweEmailAddresses) {
            String userEmail = delimeterInserter.insert(TextUtils.htmlEncode(user.getEmailAddress()), terms);
            sb.append(" - ");
            sb.append(userEmail);
        }

        sb.append(" (");
        sb.append(userName);
        sb.append(")");
        return sb.toString();
    }

    String getProjectkeyFromIssuekey(final String issuekey) {
        final int index = issuekey.indexOf('-');
        if (index < 1) { // The project key has to start with a letter
            throw new IllegalArgumentException("Somehow have an invalid issuekey: " + issuekey);
        }

        return issuekey.substring(0, index);
    }
}
