package de.accxia.jira.addon.IUM.job;


import com.atlassian.jira.security.JiraAuthenticationContext;
import com.atlassian.jira.web.session.currentusers.JiraUserSession;
import com.atlassian.plugin.spring.scanner.annotation.component.Scanned;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.status.JobDetails;
import de.accxia.jira.addon.IUM.config.DAO;
import de.accxia.jira.addon.IUM.model.JobResultDTO;
import de.accxia.jira.addon.IUM.repository.JobResultRepository;
import de.accxia.jira.addon.IUM.repository.PocketRepository;
import de.accxia.jira.addon.IUM.servlet.session.AccxiaUserSessionTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * GET/POST
 * https://aurelian-jira.accxia.com/rest/IUM/latest/job/config
 *
 *
 * https://aurelian-jira.accxia.com/rest/IUM/latest/job/start
 * https://aurelian-jira.accxia.com/rest/IUM/latest/job/stop
 *
 * https://aurelian-jira.accxia.com/rest/IUM/latest/job/runnow
 * https://aurelian-jira.accxia.com/rest/IUM/latest/job/session
 */
@Path("/job")
@Consumes({ MediaType.TEXT_HTML, MediaType.APPLICATION_JSON})
@Produces({ MediaType.APPLICATION_JSON })
@Scanned
public class JobRestController {
    private static final Logger LOG = LoggerFactory.getLogger(JobRestController.class);

    @ComponentImport
    private final JiraAuthenticationContext jiraAuthenticationContext;
    private final MonitorJobRunner monitorJobRunner;
    private final PocketRepository pocketRepository;
    private JobResultRepository jobResultRepository;
    @Inject
    public JobRestController(MonitorJobRunner monitorJobRunner,PocketRepository pocketRepository,JobResultRepository jobResultRepository,
                             JiraAuthenticationContext jiraAuthenticationContext) {
        this.monitorJobRunner=monitorJobRunner;
        this.pocketRepository = pocketRepository;
        this.jobResultRepository=jobResultRepository;
        this.jiraAuthenticationContext=jiraAuthenticationContext;
    }

    @Path("/config")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response getCurrentConfig() {
        Map<String,String> map = new HashMap<>();
        map.put(JobData.REPEAT_INTERVAL, DAO.getRepeatInterval());
        map.put(JobData.STARTING_FROM, DAO.getStartingFrom());
        map.put(JobData.ENABLE_GROUPS,DAO.getIUMGroups());
        map.put(JobData.DISABLE_GROUPS,DAO.getIUMGroupsDisabled());

        map.put("working",String.valueOf(monitorJobRunner.isWorking(IUMClusterJob.class)));
        if(monitorJobRunner.getJobDetails(IUMClusterJob.class)!=null){
            map.put("jobDetails",convert2String(monitorJobRunner.getJobDetails(IUMClusterJob.class)));
        }
        List<JobDetails> details=monitorJobRunner.getJobsByJobRunnerKey(IUMClusterJob.class);
        if(details!=null){
            for (int i = 0; i < details.size(); i++) {
                map.put("jobDetails"+i,details.get(i).toString());
            }
        }
        if(monitorJobRunner.isWorking(IUMClusterJob.class)){
            map.put("lastRunForJob",convert2String(monitorJobRunner.getLastRunForJob(IUMClusterJob.class)));
            map.put("lastSuccessfulRunForJob",convert2String(monitorJobRunner.getLastSuccessfulRunForJob(IUMClusterJob.class)));
            map.put("nextRunDate",convert2String(monitorJobRunner.getNextRunDate(IUMClusterJob.class)));
        }

        return Response.ok(map).build();
    }

    @Path("/config")
    @POST
    @Produces({ MediaType.APPLICATION_JSON })
    public Response setCurrentConfig(@QueryParam(JobData.REPEAT_INTERVAL) String repeatinterval,
                                     @QueryParam(JobData.STARTING_FROM) String startingFrom) {

        DAO.updateRepeatInterval(repeatinterval);
        DAO.updateStartingFrom(startingFrom);
        return getCurrentConfig();
    }

    @Path("/start")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response startDebugJob() {
        try{
            monitorJobRunner.doUnSchedule(IUMClusterJob.class);
            //jobKey = monitorJobRunner.doSchedule(JobDataUtils.createIUMJobData(),IUMJob.getInstance());
            //create fake schedule with running at 10 sec
            String jobKey = monitorJobRunner.doSchedule(JobDataUtils.createFakeIUMJobData(),IUMClusterJob.getInstance());

            return Response.ok("success").build();
        } catch (SchedulerServiceException e) {
            return Response.status(Response.Status.BAD_REQUEST).entity(e.getMessage()).build();
        }
    }

    @Path("/stop")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response stopDebugJob() {
        try{
            monitorJobRunner.doUnSchedule(IUMClusterJob.class);
            return Response.ok("success").build();
        } catch (SchedulerServiceException e) {
            return Response.status(Response.Status.BAD_REQUEST).entity(e.getMessage()).build();
        }
    }

    private static JobRestController.LocalThread localThread=null;

    @Path("/runnow")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response runNowsJob() {
        try {
            long duration=Long.parseLong(DAO.getInactivityDuration())*60*1000;

            Map<String, Serializable> parameters = new HashMap<>();
            parameters.put(JobData.ENABLE_GROUPS, DAO.getIUMGroups());
            parameters.put(JobData.USER_INACTIVITY_INTERVAL,duration);

            final String currentUser = jiraAuthenticationContext.getLoggedInUser() !=null ? jiraAuthenticationContext.getLoggedInUser().getUsername()  : "IUMPlugin";
            parameters.put(JobData.CURRENT_USER,currentUser );

            JobRestController.localThread = new JobRestController.LocalThread(currentUser, parameters);
            JobRestController.localThread.start();

            IUMClusterJob.getInstance().injectService(pocketRepository,jobResultRepository);
            IUMClusterJob.getInstance().doProcessingJob(parameters);

            String response = "{\"message\":\"processing\"}";
            return Response.ok(response).build();

        } catch (Exception e) {
            String response = "{\"message\":\""+e.getMessage()+"\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response ).build();
        }
    }

    @Path("/jobcheck")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response runNowsCheckJob() {
        if (JobRestController.localThread  == null) {
            String response = "{\"message\":\"not stating\"}";
            return Response.ok(response).build();
        }

        if (JobRestController.localThread.isWorking ) {
            String response = "{\"message\":\"processing\"}";
            return Response.ok(response).build();
        }

        try {

            if (JobRestController.localThread.errorMessage !=null ) {
                String response = "{\"message\":\""+JobRestController.localThread.errorMessage+"\"}";
                return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
            }

            return Response.ok(JobRestController.localThread.processResultList).build();

        } catch (Exception e) {
            String response = "{\"message\":\"" + e.getMessage() + "\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
        }
    }

    @Path("/jobresults")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response getLatestJobResults() {
        if(this.jobResultRepository == null){
            String response = "{\"message\":\"not stating\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response).build();
        }

        List<JobResultDTO> jobResultDTOList = Arrays.stream(this.jobResultRepository.findAll(0, 10))
                .map(jobResult -> JobResultDTO.fromAO(jobResult))
                .collect(Collectors.toList());

        return Response.ok(jobResultDTOList).build();
    }

    @Path("/session")
    @GET
    @Produces({ MediaType.APPLICATION_JSON })
    public Response getUserSession() {
        try {
            AccxiaUserSessionTracker accxiaUserSessionTracker= AccxiaUserSessionTracker.getInstance();
            List<JiraUserSession> userSessionList= accxiaUserSessionTracker.getAccxiaSnapshot();

            return Response.ok(userSessionList).build();
        } catch (Exception e) {
            String response = "{\"message\":\""+e.getMessage()+"\"}";
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(response ).build();
        }
    }

    class  LocalThread extends Thread {
        Map<String, Serializable> parameters;
        String currentUser;
        List<ProcessResult> processResultList=null;

        String errorMessage;
        boolean isWorking;

        public LocalThread(String currentUser,Map<String, Serializable> parameters){
            this.parameters=parameters;
            this.currentUser=currentUser;
            this.isWorking=false;
            this.errorMessage=null;
        }

        @Override
        public void run() {
            try {
                long startTime = System.currentTimeMillis();

                JobRestController.localThread.isWorking=true;
                IUMClusterJob.getInstance().injectService(pocketRepository,jobResultRepository);

                parameters.put(JobData.CURRENT_USER, currentUser );
                List<ProcessResult> processResultList = IUMClusterJob.getInstance().doProcessingJob(parameters);

                long stopTime = System.currentTimeMillis();
                long elapsedTime = stopTime - startTime;

                JobRestController.localThread.processResultList=processResultList;
                doSaveProcessResultList(processResultList,parameters,elapsedTime);

                //JobRestController.localThread.isWorking=false;
                //sendNotification(currentUser.getName(), "Automatic User Sync Job", "success");
            } catch (Exception e) {
                LOG.error("Exception: " + e.getMessage(), e);
                try {
                    JobRestController.localThread.errorMessage = e.getMessage();
                    //sendNotification(currentUser.getName(), "Automatic User Sync Job", e.getMessage());
                } catch (Exception ex) {
                    LOG.error("Exception: " + ex.getMessage(), ex);
                    throw new RuntimeException(ex);
                }
            }finally {
                JobRestController.localThread.isWorking=false;
            }
        }
    }

    private void doSaveProcessResultList(List<ProcessResult> processResultList,Map<String, Serializable> parameters, long elapsedTime) {
        final String currentUser = (String)parameters.get(JobData.CURRENT_USER);
        for (ProcessResult processResult:processResultList){
            jobResultRepository.save(JobResultDTO.createJobResult(processResult,currentUser,elapsedTime));
        }
    }

    private <T> String convert2String(T t) {
        return (t!= null) ? t.toString() : "";

    }

}
