package de.accxia.jira.addon.IUM.job;


import com.atlassian.jira.application.ApplicationRoleManager;
import com.atlassian.jira.exception.PermissionException;
import com.atlassian.jira.exception.RemoveException;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.scheduler.*;
import com.atlassian.scheduler.config.*;
import com.atlassian.scheduler.status.JobDetails;
import com.atlassian.scheduler.status.RunDetails;
import de.accxia.jira.addon.IUM.model.JobResultDTO;
import de.accxia.jira.addon.IUM.repository.JobResultRepository;
import de.accxia.jira.addon.IUM.repository.NavUserRepository;
import de.accxia.jira.addon.IUM.repository.PocketRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import java.io.Serializable;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

@ExportAsService({MonitorJobRunnerImpl.class})
@Named("monitorJobRunner")
public class MonitorJobRunnerImpl implements MonitorJobRunner {
    private Logger LOG = LoggerFactory.getLogger(MonitorJobRunnerImpl.class);

    public static final String JOB_ID = "JOB_ID";
    private final JobRunnerKey JOB_RUNNER_KEY = JobRunnerKey.of(MonitorJobRunnerImpl.class.getName());


    private final ConcurrentHashMap<JobId, JobRunnerKeyDetail> map ;
    @ComponentImport
    private final SchedulerService scheduler;
    @ComponentImport
    private final SchedulerHistoryService schedulerHistoryService;
    private final ApplicationRoleManager applicationRoleManager;
    private PocketRepository pocketRepository;
    private JobResultRepository jobResultRepository;
    private NavUserRepository navUserRepository;

    @Inject
    public MonitorJobRunnerImpl(SchedulerService scheduler, SchedulerHistoryService schedulerHistoryService,
                                ApplicationRoleManager applicationRoleManager,
                                NavUserRepository navUserRepository, PocketRepository pocketRepository,JobResultRepository jobResultRepository) {
        this.scheduler = scheduler;
        this.schedulerHistoryService = schedulerHistoryService;
        this.applicationRoleManager=applicationRoleManager;
        this.map = new ConcurrentHashMap<>();
        this.navUserRepository = navUserRepository;
        this.pocketRepository = pocketRepository;
        this.jobResultRepository=jobResultRepository;
    }

    @Override
    public String doSchedule(JobData jobData,IJob job) throws SchedulerServiceException {
        JobId jobId = JobId.of(job.getClass().getName());
        JobRunnerKeyDetail jobRunnerKeyDetail=getOrCreateJob(jobId);
        job.injectService(pocketRepository,jobResultRepository);
        job.injectService(applicationRoleManager);

        scheduler.registerJobRunner(jobRunnerKeyDetail.jobRunnerKey, new JobRunner() {
            @Nullable
            @Override
            public JobRunnerResponse runJob(JobRunnerRequest jobRequest) {
                try {
                    if (LOG.isInfoEnabled()){
                        LOG.info("Begin ProcessingJob at " + new Date());
                    }

                    doProcessingJob(job,jobRequest.getJobConfig().getParameters());

                    if (LOG.isInfoEnabled()){
                        LOG.info("End ProcessingJob at " + new Date());
                    }

                } catch (Exception e) {
                    LOG.error("Exception ProcessingJob: " + e.getMessage(),e);
                    return JobRunnerResponse.aborted("Task aborted due to the following error : " + e.getMessage());
                }
                return JobRunnerResponse.success();
            }
        });

        jobData.getParameters().put(JOB_ID,jobId.toString());
        jobRunnerKeyDetail.nextRunDate =jobData.getParameters().get(JobData.RESTART)!=null?
                createNextRunDate(jobData.getRepeatInterval(),jobData.getStartingFrom()) :
                createFirstRunDate(jobData.getRepeatInterval(),jobData.getStartingFrom());


        final JobConfig jobConfig = JobConfig
                .forJobRunnerKey(jobRunnerKeyDetail.jobRunnerKey)
                .withRunMode(RunMode.RUN_LOCALLY)
                .withParameters(jobData.getParameters())
                .withSchedule(Schedule.forInterval(jobData.getRepeatInterval(), jobRunnerKeyDetail.nextRunDate));

        try {

            scheduler.scheduleJob(jobId, jobConfig);
            LOG.warn("Job schedule with id =" + jobId.toString());
            return jobRunnerKeyDetail.jobRunnerKey.toString();
        } catch (Exception e) {
            LOG.error("Exception: " + e.getMessage(), e);
            throw new SchedulerServiceException(e.getMessage(),e);
        }
    }

    @Override
    public void doUnSchedule(Class classz)  throws SchedulerServiceException {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){

            JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);
            LOG.warn("Stopping..." + jobRunnerKeyDetail.jobRunnerKey);
            scheduler.unscheduleJob(jobId);

            getOrCreateJob(jobId).nextRunDate = null;
        }
    }

    @Override
    public JobDetails getJobDetails(Class classz)  {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            return scheduler.getJobDetails(jobId);
        }
        return null;
    }

    @Override
    public Date getNextRunDate(Class classz)  {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){
           JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);

           return jobRunnerKeyDetail.nextRunDate;
        }

        return null;
    }

    @Override
    public RunDetails getLastRunForJob(Class classz)  {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            return schedulerHistoryService.getLastRunForJob(jobId);

        }
        return null;
    }
    @Override
    public RunDetails getLastSuccessfulRunForJob(Class classz)  {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            return schedulerHistoryService.getLastSuccessfulRunForJob(jobId);
        }
        return null;
    }
    public boolean isWorking(Class classz)  {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);

            return jobRunnerKeyDetail.nextRunDate !=null;
        }
        return false;
    };

    @Override
    public List<JobDetails> getJobsByJobRunnerKey(Class classz) {
        JobId jobId= JobId.of(classz.getName());
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);

            return scheduler.getJobsByJobRunnerKey(jobRunnerKeyDetail.jobRunnerKey);
        }

        return Collections.EMPTY_LIST;
    }

    private void doProcessingJob(IJob job, Map<String, Serializable> parameters) throws PermissionException, RemoveException {

        if(job!=null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ProcessingJob parameters ");
                for (Map.Entry<String, Serializable> parameter: parameters.entrySet()) {
                    LOG.debug("key: " + parameter.getKey() + " = " + parameter.getValue());
                }
            }
            long startTime = System.currentTimeMillis();

            String className =(String) parameters.get(JOB_ID);
            JobId jobId = JobId.of(className);

            List<ProcessResult> processResultList= job.doProcessingJob(parameters);

            long stopTime = System.currentTimeMillis();
            long elapsedTime = stopTime - startTime;

            doSaveProcessResultList(processResultList,parameters,elapsedTime);
            //doEmitSystemNotification(parameters,createPlainMessage(processResultList,elapsedTime));
            Schedule schedule =scheduler.getJobDetails(jobId).getSchedule();

            if (schedule != null) {
                long timeInSecs = System.currentTimeMillis();
                Date nextRunDate =new Date(  timeInSecs + schedule.getIntervalScheduleInfo().getIntervalInMillis());
                LOG.warn("nextRunDate= " + nextRunDate) ;

                getOrCreateJob(jobId).nextRunDate = nextRunDate;
            }
        }else{
            LOG.warn("ProcessingJob NO JOB");
        }
    }

    private void doSaveProcessResultList(List<ProcessResult> processResultList,Map<String, Serializable> parameters, long elapsedTime) {
        final String currentUser = (String)parameters.get(JobData.CURRENT_USER);
        for (ProcessResult processResult:processResultList){
            jobResultRepository.save(JobResultDTO.createJobResult(processResult,currentUser,elapsedTime));
        }
    }

    private String createPlainMessage(List<ProcessResult> processResultList,long elapsedTime) {
        StringBuffer sb = new StringBuffer();
        sb.append("Status:Done").append("\n");
        for(ProcessResult processResult:processResultList){
            sb.append(String.format("Copy From %s to %s #%d",
                    processResult.getSourceGroups(),processResult.getTargetGroup(),processResult.getUserCopied())).append("\n");
            sb.append(String.format("Remove From %s #%d",
                    processResult.getTargetGroup(),processResult.getUserCopied())).append("\n");
        }

        sb.append(String.format("Duration %5.1f",elapsedTime/60.0/1000)) .append(" minutes");
        return sb.toString();
    }


    private boolean isRunning(JobId jobId){
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);
            return jobRunnerKeyDetail.nextRunDate !=null;
        }
        return false;
    }

    private boolean stopRunning(JobId jobId){
        if(map.containsKey(jobId) && map.get(jobId)!=null){
            JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);
            jobRunnerKeyDetail.nextRunDate =null;
            return true;
        }
        return false;
    }

    private JobRunnerKeyDetail getOrCreateJob(String className)  {
        JobId jobId= JobId.of(className);
        if(!map.containsKey(jobId) || map.get(jobId)==null){
            map.put(jobId, new JobRunnerKeyDetail(className));

        }

        JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);
        return jobRunnerKeyDetail;
    }
    private JobRunnerKeyDetail getOrCreateJob( JobId jobId)  {
        if(!map.containsKey(jobId) || map.get(jobId)==null){
            map.put(jobId, new JobRunnerKeyDetail(jobId.toString()));

        }

        JobRunnerKeyDetail jobRunnerKeyDetail= map.get(jobId);
        return jobRunnerKeyDetail;
    }
    private Date createFirstRunDate(long repeatInterval, int hour){
        //generate random number of seconds between 3-5 to have some gap
        Random random = new Random();
        int randomNum = (random.nextInt(5 - 3 + 1) + 3)*1000;

        //DEBUG
        if(repeatInterval <= 300000){
            long currentTimeMillis = System.currentTimeMillis();
            return  new Date(currentTimeMillis+randomNum);
        }

        Calendar calendar = Calendar.getInstance();

        //suggestion if hour is past --> then run next day
        if((calendar.get(Calendar.HOUR_OF_DAY)>=hour)){
            calendar.set(Calendar.DAY_OF_MONTH,(calendar.get(Calendar.DAY_OF_MONTH)+1));
        }

        calendar.set(Calendar.HOUR_OF_DAY, hour);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        /*
            long currentTimeMillis = System.currentTimeMillis();
            long calendarTimeMillis = calendar.getTime().getTime();
            while (calendarTimeMillis<currentTimeMillis){
                calendarTimeMillis +=repeatInterval;
            }
        */

        long calendarTimeMillis = calendar.getTime().getTime();
        return  new Date(calendarTimeMillis+randomNum);
    }

    private Date createNextRunDate(long repeatInterval, int hour){
        //generate random number of seconds between 3-5 to have some gap
        Random random = new Random();
        int randomNum = (random.nextInt(5 - 3 + 1) + 3)*1000;


        //DEBUG 3ooooo =5 min
        if(repeatInterval <= 300000){
            long currentTimeMillis = System.currentTimeMillis();
            return  new Date(currentTimeMillis+randomNum);
        }

        int dh=(int)(repeatInterval/3600000);

        Calendar calendar = Calendar.getInstance();
        int currentHour = calendar.get(Calendar.HOUR_OF_DAY);
        while (hour<23 && (hour<=currentHour)){
            hour += dh;
        }
        if(hour>23){
            //suggestion if hour is past --> then run next day
            calendar.set(Calendar.DAY_OF_MONTH,(calendar.get(Calendar.DAY_OF_MONTH)+1));
        }


        calendar.set(Calendar.HOUR_OF_DAY, hour);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        /*
            long currentTimeMillis = System.currentTimeMillis();
            long calendarTimeMillis = calendar.getTime().getTime();
            while (calendarTimeMillis<currentTimeMillis){
                calendarTimeMillis +=repeatInterval;
            }
        */

        long calendarTimeMillis = calendar.getTime().getTime();
        return  new Date(calendarTimeMillis+randomNum);
    }


    //TODO
//    private void doEmitSystemNotification( Map<String, Serializable> parameters,String body) {
//
//        final String currentUser = (String)parameters.get(JobData.CURRENT_USER);
//        try {
//            if(currentUser!=null){
//               //  sendNotification(currentUser, "Automatic User Sync Job", body);
//            }
//        } catch (Exception e) {
//            LOG.error(" "+e.getMessage(),e);
//        }
//    }

    class JobRunnerKeyDetail {
        public JobRunnerKey jobRunnerKey;
        public Date nextRunDate = null;

        public JobRunnerKeyDetail(JobRunnerKey jobRunnerKey, Date nextRunDate) {
            this.jobRunnerKey = jobRunnerKey;
            this.nextRunDate = nextRunDate;
        }

        public JobRunnerKeyDetail(JobRunnerKey jobRunnerKey) {
            this(jobRunnerKey, null);
        }

        public JobRunnerKeyDetail(String className) {
            this.jobRunnerKey = JobRunnerKey.of(className);
            this.nextRunDate=null;
        }

    }
}