let InactivityTracker = (function () {
    const INACTIVITY_LIMIT = 60000; // 1 minute
    const ACTIVITY_WINDOW = 60000; // 1 minute
    let lastActivityTime = Date.now();

    const DEBOUNCE_DELAY = 200;
    // Store debounce timers per event type
    const debounceTimers = {};

    let DEBUG=false

    function updateDebugger(value){
        DEBUG=value;
    }

    function debouncePerEvent(eventName,eventType, callback, delay) {
        return function (...args) {
            clearTimeout(debounceTimers[eventName]);
            debounceTimers[eventName] = setTimeout(() => {
                if(DEBUG){
                    console.log(`debounce eventName=${eventName} eventType=${eventType} timer= ${debounceTimers[eventName]}`  ,args) ;
                }
                callback.apply(this, args);
            }, delay);
        };
    }

    const markActivity =function() {
      if(DEBUG){
          console.log("lastActivityTime = " + lastActivityTime) ;
      }
        lastActivityTime = Date.now();
    };

    function notifyServer() {
        let url = AJS.contextPath() + "/rest/IUM/latest/apps/notifyActivity";
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ timestamp: new Date().toISOString() })
        })
            .then(response => {
                if (!response.ok) throw new Error('Server responded with error');
                console.log('Activity reported to server');
            })
            .catch(err => {
                console.error('Failed to notify server:', err);
            });
    }

    function loadConfig(){
        let url = AJS.contextPath() + "/rest/IUM/latest/apps/notifyActivityConfig";
        fetch(url, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json'
            }
        })
            .then(response => {
                if (!response.ok) throw new Error('Server responded with error');
                return response.json();
            })
            .then(data=> {
                console.log(data)
                //{activeReader: false}
                if (data && data.activeReader) {
                    setTimeout(InactivityTracker.init,5000);
                }
            })
            .catch(err => {
                console.error('Failed to notify server:', err);
            });
    }

    function attachIframeListeners(iframe) {
        try {
            const iframeDoc = iframe.contentWindow.document;
            const events = ['mousemove', 'keydown', 'scroll', 'click', 'touchstart'];

            events.forEach(eventType => {
                const handler = debouncePerEvent("iframe."+eventType,eventType, markActivity, DEBOUNCE_DELAY);
                iframeDoc.addEventListener(eventType, handler, { passive: true });
            });
        } catch (err) {
            console.warn('Could not attach listeners to iframe:', err);
        }
    }

    function waitForIframeReady(iframe) {
        try {
            //const iframeDoc = iframe.contentWindow.document;
            //console.log("Calls waitForIframeReady !",iframe);
            const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
            // Check if it's already loaded/parsed
            if (iframeDoc.readyState === "complete" || iframeDoc.readyState === "interactive") {
                if(DEBUG){
                    console.log("iframe DOM ready!");
                }
                attachIframeListeners(iframe);
            } else {
                iframeDoc.addEventListener("DOMContentLoaded", () => {
                    if(DEBUG){
                        console.log("iframe DOM ready via DOMContentLoaded!");
                    }
                    attachIframeListeners(iframe);
                });
            }
        } catch (err) {
            console.warn('Could not access iframe document:', err);
        }
    }

    function init() {
        const events = ['mousemove', 'keydown', 'scroll', 'click', 'touchstart'];

        // Attach listeners to main document
        events.forEach(eventType => {
            const handler = debouncePerEvent(eventType,eventType, markActivity, DEBOUNCE_DELAY);
            document.addEventListener(eventType, handler, { passive: true });
        });

        // Attach listeners to same-origin iframes
        const iframes = document.querySelectorAll('iframe');
        console.log("Found iframes = # " + document.querySelectorAll('iframe').length)
        iframes.forEach(iframe => {
            // console.log("Call frames = # " + document.querySelectorAll('iframe').length)
            const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
            if(iframeDoc){
                waitForIframeReady(iframe)
            }else{
                iframe.addEventListener('load', () => waitForIframeReady(iframe));
            }

        });


        setInterval(() => {
            if (Date.now() - lastActivityTime <= ACTIVITY_WINDOW) {
                notifyServer(); // 🔔 Notify once per minute if user was active
            }
        }, ACTIVITY_WINDOW);
    }



    function init2() {
        ['mousemove', 'keydown', 'scroll', 'click', 'touchstart'].forEach(event =>
            document.addEventListener(event, markActivity, { passive: true })
        );

        setInterval(() => {
            let now =Date.now();
            if (now- lastActivityTime <= ACTIVITY_WINDOW) {
                if(DEBUG){
                    console.log("User was active in the last minute.");
                }
                notifyServer(); //  Notify if user was active in the last minute
            }
        }, ACTIVITY_WINDOW);  // Check every 10 seconds

    }

    return { loadConfig, init,updateDebugger };
})();

document.addEventListener("DOMContentLoaded", () => {
    // InactivityTracker.updateDebugger(true)
    InactivityTracker.loadConfig();
});
